import {
  EffectType,
  Entity,
  EntityHealthComponent,
  EntityInitializationCause,
  EntityInventoryComponent,
  EntityIsBabyComponent,
  EntityTameableComponent,
  EntityTypeFamilyComponent,
  EntityLeashableComponent,
  ItemStack,
  system,
  world
} from '@minecraft/server';
import { CustomEntityCapturableItems } from '../item/CustomEntityCapturableItems.js';
const requiredTamed = [
  "sf_nba:hedgehog",
  "sf_nba:rat"
]
export class AbstractCapturableEntity {
  constructor(identifier, itemVariations, babyItemVariations = undefined) {
    this.identifier = identifier;
    this.itemVariations = itemVariations;
    this.babyItemVariations = babyItemVariations;
    this.isVanilla = identifier.startsWith('minecraft:');
    this.onCaptureEntity();
    this.onPlaceCapturedEntity();
  }
  onCaptureEntity() {
    world.beforeEvents.playerInteractWithEntity.subscribe(e => {
      const { player, target: entity, itemStack: stack } = e
      const ownerId = entity.getDynamicProperty('sf_nba:owner_id')
      if (!entity || !this.isCapturableEntity(entity)) return;
      if (entity.typeId == 'sf_nba:hamster') return 
      if (entity.typeId !== this.identifier) return;
      if (!stack || stack.typeId !== "sf_nba:capture_net") return;
      if (requiredTamed.includes(entity.typeId)) {
        if (ownerId) {
          if (player.id !== ownerId) return e.cancel = true
        } else {
          if (entity.getComponent('tameable')?.ownerId != player.id) return e.cancel = true
        }
      }
      system.run(() => {
        const inventory = player.getComponent(EntityInventoryComponent.componentId);
        const { dimension, location } = entity;
        if (inventory.container.emptySlotsCount === 0) {
          dimension.spawnItem(this.createEntityItem(entity), location);
        } else {
          inventory.container.addItem(this.createEntityItem(entity));
        }
        const leashable = entity.getComponent(EntityLeashableComponent.componentId);
        if (leashable?.isLeashed) leashable.unleash();
        entity.remove()
      })
    });
    world.afterEvents.playerInteractWithEntity.subscribe(e => {
      const tameComponent = e.target.getComponent('minecraft:tameable')
      if (!requiredTamed.includes(e.target.typeId)) return
      try {
        if (!tameComponent) return
        if (!tameComponent.tamedToPlayer) return
        if (!tameComponent?.isTamed || e.itemStack.typeId === 'sf_nba:capture_net') return
        e.target.setDynamicProperty('sf_nba:owner_id', tameComponent.tamedToPlayerId)
      } catch (e) { }
    })
  }
  onPlaceCapturedEntity() {
    let placedEntityData = {};
    let futureIsFirstEvent = undefined;
    system.run(() => {
      world.beforeEvents.itemUseOn.subscribe((e) => {
        const { itemStack: stack, isFirstEvent } = e;
        futureIsFirstEvent = isFirstEvent;
        if (!isFirstEvent) return;
        if (stack.typeId.startsWith('sf_nba:')) return 
        if (!stack || !stack.getTags().includes("sf_nba:captured_entity_item")) return;
        placedEntityData = {
          tick: system.currentTick,
          health: stack.getDynamicProperty("sf_nba:health"),
          ownerId: stack.getDynamicProperty("sf_nba:owner_id")
        };
      });
      world.afterEvents.entitySpawn.subscribe(({ cause, entity }) => {
        if (cause !== EntityInitializationCause.Event) return;
        if (!placedEntityData.tick) return;
        if (placedEntityData.tick !== system.currentTick || !futureIsFirstEvent) return;
        const entityHealth = entity.getComponent(EntityHealthComponent.componentId);
        entityHealth.setCurrentValue(placedEntityData.health ?? entityHealth.effectiveMax);
        placedEntityData = {};
      });
    });
  }
  createEntityItem(entity) {
    const entityHealth = entity.getComponent(EntityHealthComponent.componentId).currentValue
    const entityName = entity.nameTag;
    const tameable = entity?.getComponent(EntityTameableComponent.componentId);
    const isBaby = entity?.hasComponent(EntityIsBabyComponent.componentId);
    const variant = entity?.getProperty('sf_nba:variant') || 0;
    const dye = entity?.getProperty('sf_nba:dye') || 0;
    let itemIdentifier = !this.itemHasVariations() ? this.itemVariations : this.itemVariations.getByValue(variant) || null;
    if (isBaby && this.babyItemVariations) {
      itemIdentifier = !this.babyItemHasVariations() ? this.babyItemVariations : this.babyItemVariations.getByValue(variant) || null;
    }
    const capturableItem = new ItemStack(itemIdentifier, 1);
    if (tameable) {
      const ownerId = tameable.tamedToPlayerId;
      capturableItem.setDynamicProperty("sf_nba:owner_id", ownerId);
    }
    if (entityName) {
      capturableItem.nameTag = `§r${entityName}`;
      capturableItem.setDynamicProperty("sf_nba:name_tag", entityName);
    }
    capturableItem.setDynamicProperty("sf_nba:variant", variant);
    capturableItem.setDynamicProperty("sf_nba:dye", dye);
    capturableItem.setDynamicProperty("sf_nba:health", entityHealth);
    return capturableItem;
  }
  isCapturableEntity(entity) {
    const familyTypes = entity.getComponent(EntityTypeFamilyComponent.componentId);
    return familyTypes?.hasTypeFamily("sf_nba:capturable_entity") || this.isVanilla;
  }
  itemHasVariations() {
    return typeof this.itemVariations === 'function';
  }
  babyItemHasVariations() {
    return typeof this.babyItemVariations === 'function';
  }
}